<?php
header('Content-Type: text/html; charset=UTF-8');
mb_internal_encoding('UTF-8');
mb_http_output('UTF-8');
require_once __DIR__ . '/../../src/config.php';
require_once __DIR__ . '/../../src/csrf.php';
require_once __DIR__ . '/../../src/layout.php';
require_login();
$user = current_user();
if ($user['user_type'] !== 'teacher') { http_response_code(403); echo 'Forbidden'; exit; }

$paperId = (int)($_GET['paper_id'] ?? 0);
if (!$paperId) { echo 'Missing paper'; exit; }
$pdo = db();

// Fetch paper and questions
$pStmt = $pdo->prepare('SELECT id, title, fee_cents, time_limit_seconds, is_published FROM papers WHERE id=? AND teacher_id=?');
$pStmt->execute([$paperId, $user['id']]);
$paper = $pStmt->fetch();
if (!$paper) { echo 'Paper not found'; exit; }

$qStmt = $pdo->prepare('SELECT id, question_text, marks, position, image_path FROM questions WHERE paper_id=? ORDER BY position');
$qStmt->execute([$paperId]);
$questions = $qStmt->fetchAll();
$oStmt = $pdo->prepare('SELECT id, option_text, is_correct FROM answer_options WHERE question_id=? ORDER BY id');
foreach ($questions as &$q) {
    $oStmt->execute([$q['id']]);
    $q['options'] = $oStmt->fetchAll();
}
unset($q);

$errors = [];
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!csrf_verify()) {
        $errors[] = 'Invalid CSRF token';
    }
    $title = trim($_POST['title'] ?? '');
    $fee = (int)($_POST['fee_cents'] ?? 0);
    $timeLimit = (int)($_POST['time_limit_minutes'] ?? 0) * 60;
    $publish = isset($_POST['publish']) ? 1 : 0;
    $questionsIn = $_POST['questions'] ?? [];
    $optionsIn = $_POST['options'] ?? [];
    $correctIn = $_POST['correct'] ?? [];
    $marksIn = $_POST['marks'] ?? [];

    if (!$title || $timeLimit <= 0) { $errors[] = 'Title and time limit required.'; }
    if (empty($questionsIn)) { $errors[] = 'At least one question required.'; }

    // Server-side correct answer validation
    foreach ($questionsIn as $i => $qText) {
        $opts = $optionsIn[$i] ?? [];
        if (empty($opts)) { $errors[] = 'Each question needs options.'; break; }
        if (!isset($correctIn[$i])) { $errors[] = 'Select a correct option for every question.'; break; }
    }

    if (!$errors) {
        try {
            $pdo->beginTransaction();
            $u = $pdo->prepare('UPDATE papers SET title=?, fee_cents=?, time_limit_seconds=?, is_published=? WHERE id=? AND teacher_id=?');
            $u->execute([$title, $fee, $timeLimit, $publish, $paperId, $user['id']]);

            // Remove old questions/options
            $pdo->prepare('DELETE FROM questions WHERE paper_id=?')->execute([$paperId]);

            // Reinsert questions and options
            $qIns = $pdo->prepare('INSERT INTO questions (paper_id, question_text, marks, position) VALUES (?,?,?,?)');
            $oIns = $pdo->prepare('INSERT INTO answer_options (question_id, option_text, is_correct) VALUES (?,?,?)');
            foreach ($questionsIn as $i => $qText) {
                $qText = trim($qText);
                if ($qText === '') { continue; }
                $marks = (int)($marksIn[$i] ?? 1); if ($marks < 1) { $marks = 1; }
                $qIns->execute([$paperId, $qText, $marks, $i + 1]);
                $newQid = $pdo->lastInsertId();
                $opts = $optionsIn[$i] ?? [];
                foreach ($opts as $j => $optText) {
                    $isCorrect = ((string)($correctIn[$i] ?? '') === (string)$j) ? 1 : 0;
                    $oIns->execute([$newQid, trim($optText), $isCorrect]);
                }
            }

            $pdo->commit();
            $success = 'Paper updated.';
            // Refresh data for display
            header('Location: ' . app_href('teacher/edit_paper.php?paper_id=' . $paperId . '&saved=1'));
            exit;
        } catch (Throwable $e) {
            $pdo->rollBack();
            $errors[] = 'Error updating paper: ' . $e->getMessage();
        }
    }
}

$paperJson = json_encode($paper);
$questionsJson = json_encode($questions);
render_header('Edit Paper');
?>

<!-- Load jQuery first for MathQuill -->
<script src="https://ajax.googleapis.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>

<!-- MathQuill CSS & JS for visual math editor -->
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/mathquill/0.10.1/mathquill.min.css" crossorigin="anonymous" />
<script src="https://cdnjs.cloudflare.com/ajax/libs/mathquill/0.10.1/mathquill.min.js" crossorigin="anonymous"></script>

<!-- Load KaTeX for math rendering -->
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/katex@0.16.9/dist/katex.min.css" />
<script src="https://cdn.jsdelivr.net/npm/katex@0.16.9/dist/katex.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/katex@0.16.9/dist/contrib/auto-render.min.js"></script>

<script>
  // Initialize KaTeX rendering when script loads
  function renderMath() {
    if (window.renderMathInElement) {
      try {
        window.renderMathInElement(document.body, {
          delimiters: [
            {left: '$$', right: '$$', display: true},
            {left: '$', right: '$', display: false},
            {left: '\\[', right: '\\]', display: true},
            {left: '\\(', right: '\\)', display: false}
          ],
          throwOnError: false
        });
      } catch(err) {
        console.log('KaTeX render error:', err.message);
      }
    }
  }
  
  // Render when DOM is ready
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', renderMath);
  } else {
    renderMath();
  }
</script>

<section class="mb-4">
  <div class="d-flex flex-wrap justify-content-between align-items-center gap-3 mb-2">
    <div>
      <p class="text-uppercase small text-muted mb-1">Editing Paper</p>
      <h1 class="h4 mb-0"><?= htmlspecialchars($paper['title']) ?></h1>
    </div>
    <div class="d-flex gap-2">
      <a class="btn btn-outline-secondary" href="<?= htmlspecialchars(app_href('teacher/manage_papers.php')) ?>">Back to list</a>
    </div>
  </div>
</section>
<section class="app-card p-4">
  <div class="d-flex flex-wrap justify-content-between align-items-center gap-3 mb-3">
    <h2 class="h5 mb-0">Paper Details</h2>
    <span class="badge-soft <?= $paper['is_published'] ? 'success' : 'muted' ?>"><?= $paper['is_published'] ? 'Published' : 'Draft' ?></span>
  </div>
  <?php if (isset($_GET['saved'])): ?><div class="alert alert-success">Saved successfully.</div><?php endif; ?>
  <?php foreach ($errors as $e): ?><div class="alert alert-danger" role="alert" aria-live="assertive"><?= htmlspecialchars($e) ?></div><?php endforeach; ?>
  <?php if ($success): ?><div class="alert alert-success" role="status" aria-live="polite"><?= htmlspecialchars($success) ?></div><?php endif; ?>
  <form method="post" id="editForm">
    <?= csrf_field(); ?>
    <div class="row g-3">
      <div class="col-lg-6">
        <label class="form-label">Title</label>
        <input class="form-control" name="title" id="title" required>
      </div>
      <div class="col-6 col-lg-3">
        <label class="form-label">Fee (cents)</label>
        <input type="number" class="form-control" name="fee_cents" id="fee_cents" min="0" value="0">
      </div>
      <div class="col-6 col-lg-3">
        <label class="form-label">Time Limit (minutes)</label>
        <input type="number" class="form-control" name="time_limit_minutes" id="time_limit_minutes" min="1" required>
      </div>
      <div class="col-12 form-check ps-2">
        <input type="checkbox" class="form-check-input" id="publishNow" name="publish" value="1">
        <label class="form-check-label" for="publishNow">Publish immediately</label>
      </div>
    </div>
    <hr class="my-4">
    <div class="d-flex justify-content-between align-items-center mb-3">
      <h3 class="h6 mb-0">Questions</h3>
      <button type="button" class="btn btn-primary btn-sm d-inline-flex align-items-center gap-2" onclick="addQuestion()"><i class="bi bi-plus-circle"></i> Add Question</button>
    </div>
    <div id="questions" class="d-flex flex-column gap-3" aria-live="polite"></div>
    <div class="d-flex flex-wrap gap-2 mt-4">
      <button type="submit" class="btn btn-primary d-inline-flex align-items-center gap-2"><i class="bi bi-save"></i> Save Changes</button>
      <button type="button" class="btn btn-outline-secondary" onclick="addQuestion()">Add Question</button>
    </div>
    <p class="muted small mt-3 mb-0"><strong>Reminder:</strong> You must select the correct answer using the radio button next to it. Submissions without a marked correct answer will be blocked.</p>
  </form>
</section>
<script>
const paperData = <?= $paperJson ?>;
const questionsData = <?= $questionsJson ?>;
let qIndex = 0;

function updateQuestionCount(){
  const hdr = document.getElementById('questionCount');
  if (hdr) { hdr.textContent = document.querySelectorAll('.question-card').length; }
}

function addQuestion(prefill){
  const wrap = document.createElement('div');
  wrap.className = 'question-card border rounded-3 p-3 bg-white shadow-sm';
  const idx = qIndex;
  wrap.innerHTML = `
    <div class="d-flex justify-content-between align-items-center mb-2">
      <h4 class="h6 mb-0">Question #${idx + 1}</h4>
      <button type="button" class="btn btn-sm btn-outline-danger" onclick="removeQuestion(this)"><i class="bi bi-x"></i></button>
    </div>
    <label class="form-label">Question text</label>
    <textarea class="form-control mb-3" name="questions[${idx}]" rows="3" required></textarea>
    <div class="mb-3">
      <label class="form-label">Marks</label>
      <input type="number" class="form-control" name="marks[${idx}]" min="1" value="1" required>
    </div>
    <div class="mb-3">
      <label class="form-label">Options</label>
      <div class="d-flex flex-column gap-2" id="opts_${idx}"></div>
      <button type="button" class="btn btn-outline-primary btn-sm mt-2" onclick="addOption(${idx})"><i class="bi bi-plus"></i> Add option</button>
    </div>
    <div class="alert alert-warning small mb-0" role="alert">
      <i class="bi bi-exclamation-triangle-fill"></i> <strong>Important:</strong> You must select which option is correct.
    </div>
  `;
  document.getElementById('questions').appendChild(wrap);
  qIndex++;
  updateQuestionCount();

  // Prefill content
  if (prefill) {
    wrap.querySelector('textarea').value = prefill.question_text || '';
    wrap.querySelector('input[name="marks['+idx+']"]').value = prefill.marks || 1;
    const opts = prefill.options || [];
    if (opts.length === 0) { addOption(idx); addOption(idx); }
    opts.forEach((o, j) => addOption(idx, o, j));
  } else {
    addOption(idx); addOption(idx);
  }
}

function removeQuestion(btn){
  btn.closest('.question-card').remove();
  updateQuestionCount();
}

function addOption(i, prefill, optIndex){
  const oWrap = document.getElementById('opts_' + i);
  const count = (typeof optIndex === 'number') ? optIndex : oWrap.querySelectorAll('.input-group').length;
  const div = document.createElement('div');
  div.className = 'input-group align-items-center';
  div.innerHTML = `
    <input type="radio" class="form-check-input ms-2" name="correct[${i}]" value="${count}" id="correct_${i}_${count}">
    <input type="text" class="form-control" name="options[${i}][${count}]" placeholder="Option text" required>
    <button class="btn btn-outline-danger" type="button" onclick="this.closest('.input-group').remove()">Remove</button>
  `;
  oWrap.appendChild(div);
  if (prefill) {
    div.querySelector('input[type=text]').value = prefill.option_text;
    if (prefill.is_correct) { div.querySelector('input[type=radio]').checked = true; }
  }
}

document.addEventListener('DOMContentLoaded', () => {
  // Prefill paper fields
  document.getElementById('title').value = paperData.title || '';
  document.getElementById('fee_cents').value = paperData.fee_cents || 0;
  document.getElementById('time_limit_minutes').value = Math.max(1, Math.round((paperData.time_limit_seconds||60)/60));
  if (paperData.is_published) { document.getElementById('publishNow').checked = true; }

  if (Array.isArray(questionsData) && questionsData.length) {
    questionsData.forEach(q => addQuestion(q));
  } else {
    addQuestion();
  }

  // Validate correct answers before submit
  document.getElementById('editForm').addEventListener('submit', (e) => {
    const cards = document.querySelectorAll('.question-card');
    let hasError = false;
    cards.forEach((card, idx) => {
      const radios = card.querySelectorAll(`input[name="correct[${idx}]"]`);
      const isChecked = Array.from(radios).some(r => r.checked);
      if (!isChecked && radios.length > 0) {
        hasError = true;
        const alertBox = card.querySelector('.alert');
        if (alertBox) {
          alertBox.classList.remove('alert-warning');
          alertBox.classList.add('alert-danger');
          alertBox.innerHTML = '<i class="bi bi-exclamation-triangle-fill"></i> <strong>ERROR:</strong> Select the correct option.';
        }
      }
    });
    if (hasError) {
      e.preventDefault();
      alert('Please select the correct answer for every question.');
      window.scrollTo({ top: 0, behavior: 'smooth' });
    }
  });
});
</script>
<?php render_footer(); ?>
